// Biometric Fingerprint Attendance System - Main JavaScript File
document.addEventListener('DOMContentLoaded', function() {
    // DOM Elements
    const attendanceScreen = document.getElementById('attendanceScreen');
    const registrationScreen = document.getElementById('registrationScreen');
    const adminLoginScreen = document.getElementById('adminLoginScreen');
    const scanPrompt = document.getElementById('scanPrompt');
    const scanningIndicator = document.getElementById('scanningIndicator');
    const resultMessage = document.getElementById('resultMessage');
    const registerBtn = document.getElementById('registerBtn');
    const adminBtn = document.getElementById('adminBtn');
    const userTypeSelect = document.getElementById('userType');
    const studentFields = document.getElementById('studentFields');
    const registrationForm = document.getElementById('registrationForm');
    const adminLoginForm = document.getElementById('adminLoginForm');
    const cancelRegistration = document.getElementById('cancelRegistration');
    const backFromAdmin = document.getElementById('backFromAdmin');

    // State variables
    let currentUserType = null;
    let isScanning = false;

    // Initialize the application
    initApp();

    function initApp() {
        // Set up event listeners
        registerBtn.addEventListener('click', showRegistrationScreen);
        adminBtn.addEventListener('click', showAdminLoginScreen);
        userTypeSelect.addEventListener('change', toggleStudentFields);
        registrationForm.addEventListener('submit', handleRegistration);
        adminLoginForm.addEventListener('submit', handleAdminLogin);
        cancelRegistration.addEventListener('click', showAttendanceScreen);
        backFromAdmin.addEventListener('click', showAttendanceScreen);

        // Show initial attendance screen
        showAttendanceScreen();
    }

    // Show attendance screen
    function showAttendanceScreen() {
        attendanceScreen.classList.add('active');
        attendanceScreen.classList.remove('hidden');
        
        registrationScreen.classList.remove('active');
        registrationScreen.classList.add('hidden');
        
        adminLoginScreen.classList.remove('active');
        adminLoginScreen.classList.add('hidden');
        
        clearResultMessage();
    }

    // Show registration screen
    function showRegistrationScreen() {
        attendanceScreen.classList.remove('active');
        attendanceScreen.classList.add('hidden');
        
        registrationScreen.classList.add('active');
        registrationScreen.classList.remove('hidden');
        
        adminLoginScreen.classList.remove('active');
        adminLoginScreen.classList.add('hidden');
        
        clearResultMessage();
    }

    // Show admin login screen
    function showAdminLoginScreen() {
        attendanceScreen.classList.remove('active');
        attendanceScreen.classList.add('hidden');
        
        registrationScreen.classList.remove('active');
        registrationScreen.classList.add('hidden');
        
        adminLoginScreen.classList.add('active');
        adminLoginScreen.classList.remove('hidden');
        
        clearResultMessage();
    }

    // Toggle student fields based on user type selection
    function toggleStudentFields() {
        const selectedValue = userTypeSelect.value;
        if (selectedValue === 'student') {
            studentFields.classList.remove('hidden');
        } else {
            studentFields.classList.add('hidden');
        }
    }

    // Handle user registration
    async function handleRegistration(e) {
        e.preventDefault();
        
        const userId = document.getElementById('userId').value;
        const userName = document.getElementById('userName').value;
        const userType = document.getElementById('userType').value;
        
        // Collect student-specific fields if applicable
        let className = '', section = '', rollNumber = '';
        if (userType === 'student') {
            className = document.getElementById('className').value;
            section = document.getElementById('section').value;
            rollNumber = document.getElementById('rollNumber').value;
        }
        
        // Validate inputs
        if (!userId || !userName || !userType) {
            showResultMessage('Please fill in all required fields.', 'error');
            return;
        }
        
        // Simulate fingerprint registration process
        showResultMessage('Placing finger on scanner...', 'warning');
        
        // Simulate scanning delay
        setTimeout(async () => {
            try {
                // Simulate API call to register user
                const userData = {
                    id: userId,
                    name: userName,
                    type: userType,
                    class: userType === 'student' ? className : null,
                    section: userType === 'student' ? section : null,
                    rollNumber: userType === 'student' ? rollNumber : null
                };
                
                const response = await simulateApiCall('/api/register', userData);
                
                if (response.success) {
                    showResultMessage('User registered successfully!', 'success');
                    
                    // Clear form
                    registrationForm.reset();
                    studentFields.classList.add('hidden');
                    
                    // Return to attendance screen after delay
                    setTimeout(() => {
                        showAttendanceScreen();
                    }, 2000);
                } else {
                    showResultMessage(response.message || 'Registration failed. Please try again.', 'error');
                }
            } catch (error) {
                showResultMessage('Registration failed. Please try again.', 'error');
            }
        }, 2000);
    }

    // Handle admin login
    async function handleAdminLogin(e) {
        e.preventDefault();
        
        const username = document.getElementById('adminUsername').value;
        const password = document.getElementById('adminPassword').value;
        
        if (!username || !password) {
            showResultMessage('Please enter both username and password.', 'error');
            return;
        }
        
        try {
            // Simulate API call to authenticate admin
            const response = await simulateApiCall('/api/admin/login', { username, password });
            
            if (response.success) {
                showResultMessage('Login successful! Redirecting...', 'success');
                
                // Clear form
                adminLoginForm.reset();
                
                // In a real app, redirect to admin panel
                setTimeout(() => {
                    alert('Admin panel would open here. This is a demo.');
                    showAttendanceScreen();
                }, 1500);
            } else {
                showResultMessage(response.message || 'Invalid credentials. Please try again.', 'error');
            }
        } catch (error) {
            showResultMessage('Login failed. Please try again.', 'error');
        }
    }

    // Simulate fingerprint scanning for attendance
    async function simulateFingerprintScan() {
        if (isScanning) return;
        
        isScanning = true;
        
        // Update UI to show scanning state
        scanPrompt.textContent = 'Scanning... Please wait.';
        scanningIndicator.classList.remove('hidden');
        clearResultMessage();
        
        try {
            // Simulate scanning delay
            await new Promise(resolve => setTimeout(resolve, 3000));
            
            // Generate random result (success or failure)
            const isSuccess = Math.random() > 0.3; // 70% success rate for demo
            
            if (isSuccess) {
                // Simulate successful scan
                const response = await simulateApiCall('/api/attendance/mark', {
                    timestamp: new Date().toISOString()
                });
                
                if (response.success) {
                    scanPrompt.textContent = 'Please place your finger on the scanner.';
                    scanningIndicator.classList.add('hidden');
                    showResultMessage('Attendance recorded successfully!', 'success');
                    isScanning = false;
                } else {
                    scanPrompt.textContent = 'Please place your finger on the scanner.';
                    scanningIndicator.classList.add('hidden');
                    showResultMessage(response.message || 'Failed to record attendance.', 'error');
                    isScanning = false;
                }
            } else {
                // Simulate failed scan
                scanPrompt.textContent = 'Please place your finger on the scanner.';
                scanningIndicator.classList.add('hidden');
                showResultMessage('Fingerprint not recognized. Please try again.', 'error');
                isScanning = false;
            }
        } catch (error) {
            scanPrompt.textContent = 'Please place your finger on the scanner.';
            scanningIndicator.classList.add('hidden');
            showResultMessage('Scanning failed. Please try again.', 'error');
            isScanning = false;
        }
    }

    // Show result message with appropriate styling
    function showResultMessage(message, type) {
        resultMessage.textContent = message;
        resultMessage.className = `result-message ${type} fade-in`;
        
        // Auto-hide success messages after 3 seconds
        if (type === 'success') {
            setTimeout(() => {
                clearResultMessage();
            }, 3000);
        }
    }

    // Clear result message
    function clearResultMessage() {
        resultMessage.textContent = '';
        resultMessage.className = 'result-message';
    }

    // Simulate API call (in a real app, this would be actual HTTP request)
    function simulateApiCall(endpoint, data) {
        return new Promise((resolve) => {
            setTimeout(() => {
                // Simulate different responses based on endpoint
                if (endpoint.includes('/register')) {
                    // Simulate 90% success rate for registration
                    const success = Math.random() > 0.1;
                    resolve({
                        success: success,
                        message: success ? 'User registered successfully' : 'Registration failed. User ID may already exist.'
                    });
                } else if (endpoint.includes('/login')) {
                    // Simulate 80% success rate for login (valid credentials: admin/admin)
                    const isValid = data.username === 'admin' && data.password === 'admin';
                    resolve({
                        success: isValid,
                        message: isValid ? 'Login successful' : 'Invalid credentials'
                    });
                } else if (endpoint.includes('/mark')) {
                    // Simulate attendance marking
                    // Check if already marked today (for demo purposes)
                    const alreadyMarked = Math.random() > 0.7; // 30% chance of duplicate
                    if (alreadyMarked) {
                        resolve({
                            success: false,
                            message: 'Attendance already marked today.'
                        });
                    } else {
                        resolve({
                            success: true,
                            message: 'Attendance recorded successfully',
                            timestamp: new Date().toISOString()
                        });
                    }
                } else {
                    // Default success response
                    resolve({ success: true });
                }
            }, 500); // Simulate network delay
        });
    }

    // Set up click listener for the fingerprint scanner area to trigger scanning
    const fingerprintScanners = document.querySelectorAll('.fingerprint-scanner:not(.small)');
    fingerprintScanners.forEach(scanner => {
        scanner.addEventListener('click', () => {
            if (!isScanning) {
                simulateFingerprintScan();
            }
        });
    });

    // Add touch feedback for mobile devices
    fingerprintScanners.forEach(scanner => {
        scanner.addEventListener('touchstart', () => {
            scanner.style.transform = 'scale(0.95)';
        });
        
        scanner.addEventListener('touchend', () => {
            scanner.style.transform = 'scale(1)';
        });
    });

    // Demo mode: Allow clicking anywhere on the scanner to simulate scan
    console.log('Biometric Attendance System initialized. Click on the scanner to simulate fingerprint scan.');
});