// Admin Panel JavaScript
document.addEventListener('DOMContentLoaded', function() {
    // DOM Elements
    const navLinks = document.querySelectorAll('.nav-link');
    const contentSections = document.querySelectorAll('.content-section');
    const logoutBtn = document.getElementById('logoutBtn');
    const addUserBtn = document.getElementById('addUserBtn');
    const userModal = document.getElementById('userModal');
    const closeModal = document.querySelector('.close');
    const cancelUserBtn = document.getElementById('cancelUserBtn');
    const userForm = document.getElementById('userForm');
    const userTypeSelect = document.getElementById('userType');
    const studentFields = document.getElementById('studentFields');
    const modalTitle = document.getElementById('modalTitle');
    const attendanceDateInput = document.getElementById('attendanceDate');
    const reportMonthInput = document.getElementById('reportMonth');
    const exportReportBtn = document.getElementById('exportReportBtn');
    
    // Initialize date inputs
    const today = new Date().toISOString().split('T')[0];
    attendanceDateInput.value = today;
    
    const currentMonth = new Date().toISOString().slice(0, 7);
    reportMonthInput.value = currentMonth;

    // Initialize the admin panel
    initAdminPanel();

    function initAdminPanel() {
        // Set up navigation
        navLinks.forEach(link => {
            link.addEventListener('click', function(e) {
                if (this.id === 'logoutBtn') return;
                
                e.preventDefault();
                
                // Remove active class from all links and sections
                navLinks.forEach(l => l.classList.remove('active'));
                contentSections.forEach(s => {
                    s.classList.remove('active');
                    s.classList.add('hidden');
                });
                
                // Add active class to clicked link
                this.classList.add('active');
                
                // Show corresponding section
                const sectionId = this.getAttribute('data-section');
                const section = document.getElementById(sectionId);
                if (section) {
                    section.classList.remove('hidden');
                    section.classList.add('active');
                    
                    // Load content based on section
                    switch(sectionId) {
                        case 'dashboard':
                            loadDashboardData();
                            break;
                        case 'manage-users':
                            loadUsersData();
                            break;
                        case 'daily-attendance':
                            loadAttendanceData();
                            break;
                        case 'monthly-reports':
                            loadReportsData();
                            break;
                    }
                }
            });
        });

        // Set up modal events
        addUserBtn.addEventListener('click', () => {
            modalTitle.textContent = 'Add New User';
            userForm.reset();
            studentFields.classList.add('hidden');
            userModal.classList.remove('hidden');
        });

        closeModal.addEventListener('click', () => {
            userModal.classList.add('hidden');
        });

        cancelUserBtn.addEventListener('click', () => {
            userModal.classList.add('hidden');
        });

        window.addEventListener('click', (e) => {
            if (e.target === userModal) {
                userModal.classList.add('hidden');
            }
        });

        // Set up form events
        userTypeSelect.addEventListener('change', toggleStudentFields);
        userForm.addEventListener('submit', handleUserSubmit);
        logoutBtn.addEventListener('click', handleLogout);
        attendanceDateInput.addEventListener('change', loadAttendanceData);
        reportMonthInput.addEventListener('change', loadReportsData);
        exportReportBtn.addEventListener('click', exportReport);

        // Load dashboard data initially
        loadDashboardData();
    }

    // Toggle student fields based on user type
    function toggleStudentFields() {
        const selectedValue = userTypeSelect.value;
        if (selectedValue === 'student') {
            studentFields.classList.remove('hidden');
        } else {
            studentFields.classList.add('hidden');
        }
    }

    // Handle user form submission
    async function handleUserSubmit(e) {
        e.preventDefault();
        
        const userId = document.getElementById('userId').value;
        const userName = document.getElementById('userName').value;
        const userType = document.getElementById('userType').value;
        const className = document.getElementById('className').value;
        const sectionName = document.getElementById('sectionName').value;
        const rollNumber = document.getElementById('rollNumber').value;
        
        // Validate inputs
        if (!userId || !userName || !userType) {
            alert('Please fill in all required fields.');
            return;
        }
        
        try {
            const userData = {
                id: userId,
                name: userName,
                type: userType,
                class: userType === 'student' ? className : null,
                section: userType === 'student' ? sectionName : null,
                rollNumber: userType === 'student' ? rollNumber : null
            };
            
            // In a real app, this would be an API call
            // For demo, we'll simulate the API call
            const response = await simulateApiCall('/api/register', userData, 'POST');
            
            if (response.success) {
                alert('User added successfully!');
                userModal.classList.add('hidden');
                loadUsersData(); // Refresh the user list
            } else {
                alert(response.message || 'Failed to add user. Please try again.');
            }
        } catch (error) {
            alert('Error adding user: ' + error.message);
        }
    }

    // Handle logout
    async function handleLogout() {
        try {
            // In a real app, this would be an API call
            const response = await simulateApiCall('/api/admin/logout', {}, 'POST');
            
            if (response.success) {
                window.location.href = 'index.html'; // Redirect to login page
            } else {
                alert('Logout failed. Please try again.');
            }
        } catch (error) {
            alert('Error during logout: ' + error.message);
        }
    }

    // Load dashboard data
    async function loadDashboardData() {
        try {
            // In a real app, this would be an API call
            const statsResponse = await simulateApiCall('/api/stats', {}, 'GET');
            
            if (statsResponse.success) {
                document.getElementById('totalUsers').textContent = statsResponse.stats.total_users || 0;
                document.getElementById('todayAttendance').textContent = statsResponse.stats.today_attendance || 0;
                document.getElementById('totalAttendance').textContent = statsResponse.stats.total_attendance || 0;
                document.getElementById('activeAdmins').textContent = statsResponse.stats.total_admins || 0;
            }
            
            // Load recent activity
            loadRecentActivity();
        } catch (error) {
            console.error('Error loading dashboard data:', error);
        }
    }

    // Load recent activity
    function loadRecentActivity() {
        const activityList = document.getElementById('activityList');
        activityList.innerHTML = `
            <div class="activity-item">
                <strong>New user registered:</strong> John Doe (Student) - Today 10:30 AM
            </div>
            <div class="activity-item">
                <strong>Attendance marked:</strong> Jane Smith - Today 9:15 AM
            </div>
            <div class="activity-item">
                <strong>Attendance marked:</strong> Robert Johnson - Today 9:05 AM
            </div>
            <div class="activity-item">
                <strong>User updated:</strong> Emily Davis (Teacher) - Yesterday 4:45 PM
            </div>
            <div class="activity-item">
                <strong>System backup completed:</strong> - Yesterday 11:59 PM
            </div>
        `;
    }

    // Load users data
    async function loadUsersData() {
        try {
            // In a real app, this would be an API call
            const response = await simulateApiCall('/api/users', {}, 'GET');
            
            if (response.success) {
                const users = response.users || [];
                renderUsersTable(users);
            } else {
                throw new Error(response.message || 'Failed to load users');
            }
        } catch (error) {
            console.error('Error loading users:', error);
            document.getElementById('usersTableBody').innerHTML = `
                <tr><td colspan="6">Error loading users: ${error.message}</td></tr>
            `;
        }
    }

    // Render users table
    function renderUsersTable(users) {
        const tbody = document.getElementById('usersTableBody');
        
        if (users.length === 0) {
            tbody.innerHTML = '<tr><td colspan="6">No users found</td></tr>';
            return;
        }
        
        tbody.innerHTML = users.map(user => `
            <tr>
                <td>${user.user_id || 'N/A'}</td>
                <td>${user.name || 'N/A'}</td>
                <td>${user.user_type || 'N/A'}</td>
                <td>${user.class_name || 'N/A'}</td>
                <td>${user.section_name || 'N/A'}</td>
                <td>
                    <div class="action-buttons">
                        <button class="action-btn edit-btn" onclick="editUser('${user.user_id}')">Edit</button>
                        <button class="action-btn delete-btn" onclick="deleteUser('${user.user_id}')">Delete</button>
                    </div>
                </td>
            </tr>
        `).join('');
    }

    // Load attendance data
    async function loadAttendanceData() {
        try {
            const date = attendanceDateInput.value;
            
            // In a real app, this would be an API call
            const response = await simulateApiCall(`/api/attendance/daily-summary?date=${date}`, {}, 'GET');
            
            if (response.success) {
                const summary = response.summary || [];
                renderAttendanceTable(summary);
                
                // Update summary counts
                document.getElementById('totalPresent').textContent = summary.length;
                document.getElementById('lateEntries').textContent = summary.filter(item => item.is_late).length;
                // For demo, we'll calculate absent as 50 - present (this is just for demo)
                document.getElementById('absentCount').textContent = Math.max(0, 50 - summary.length);
            } else {
                throw new Error(response.message || 'Failed to load attendance data');
            }
        } catch (error) {
            console.error('Error loading attendance data:', error);
            document.getElementById('attendanceTableBody').innerHTML = `
                <tr><td colspan="6">Error loading attendance data: ${error.message}</td></tr>
            `;
        }
    }

    // Render attendance table
    function renderAttendanceTable(attendanceData) {
        const tbody = document.getElementById('attendanceTableBody');
        
        if (attendanceData.length === 0) {
            tbody.innerHTML = '<tr><td colspan="6">No attendance data for selected date</td></tr>';
            return;
        }
        
        tbody.innerHTML = attendanceData.map(record => `
            <tr>
                <td>${record.name || 'N/A'}</td>
                <td>${record.user_id || 'N/A'}</td>
                <td>${record.type_name || 'N/A'}</td>
                <td>${record.first_scan ? new Date(record.first_scan).toLocaleTimeString() : 'N/A'}</td>
                <td>${record.last_scan ? new Date(record.last_scan).toLocaleTimeString() : 'N/A'}</td>
                <td>
                    ${record.is_late ? 
                        '<span style="color: #e74c3c; font-weight: bold;">Late</span>' : 
                        '<span style="color: #2ecc71; font-weight: bold;">On Time</span>'
                    }
                </td>
            </tr>
        `).join('');
    }

    // Load reports data
    async function loadReportsData() {
        try {
            const monthYear = reportMonthInput.value;
            const [year, month] = monthYear.split('-');
            
            // In a real app, this would be an API call
            const response = await simulateApiCall(`/api/attendance/monthly-report?month=${month}&year=${year}`, {}, 'GET');
            
            if (response.success) {
                const report = response.report || [];
                renderReportsTable(report);
                
                // Initialize charts
                initializeCharts(report);
            } else {
                throw new Error(response.message || 'Failed to load reports');
            }
        } catch (error) {
            console.error('Error loading reports:', error);
            document.getElementById('reportsTableBody').innerHTML = `
                <tr><td colspan="5">Error loading reports: ${error.message}</td></tr>
            `;
        }
    }

    // Render reports table
    function renderReportsTable(reports) {
        const tbody = document.getElementById('reportsTableBody');
        
        if (reports.length === 0) {
            tbody.innerHTML = '<tr><td colspan="5">No reports for selected month</td></tr>';
            return;
        }
        
        tbody.innerHTML = reports.map(record => {
            // Calculate attendance percentage (for demo)
            const totalDays = 30; // Assuming 30 days in month
            const attendancePercentage = totalDays > 0 ? 
                Math.round((record.days_present / totalDays) * 100) : 0;
                
            return `
            <tr>
                <td>${record.name || 'N/A'}</td>
                <td>${record.user_id || 'N/A'}</td>
                <td>${record.type_name || 'N/A'}</td>
                <td>${record.days_present || 0}</td>
                <td>${attendancePercentage}%</td>
            </tr>
        `}).join('');
    }

    // Initialize charts (for demo purposes)
    function initializeCharts(data) {
        // Note: In a real app, you would use Chart.js or similar library
        console.log('Initializing charts with data:', data);
        
        // For demo, we'll just log that charts would be initialized
        // In a real implementation, you would create actual charts here
    }

    // Export report
    function exportReport() {
        alert('Report export functionality would be implemented here.\nIn a real app, this would generate an Excel/PDF report.');
    }

    // Edit user function (global for use in HTML)
    window.editUser = function(userId) {
        // In a real app, this would fetch user details and populate the form
        alert(`Edit user functionality would be implemented here for user ID: ${userId}`);
    };

    // Delete user function (global for use in HTML)
    window.deleteUser = async function(userId) {
        if (!confirm(`Are you sure you want to delete user with ID: ${userId}?`)) {
            return;
        }
        
        try {
            // In a real app, this would be an API call
            const response = await simulateApiCall(`/api/users/${userId}`, {}, 'DELETE');
            
            if (response.success) {
                alert('User deleted successfully!');
                loadUsersData(); // Refresh the user list
            } else {
                alert(response.message || 'Failed to delete user. Please try again.');
            }
        } catch (error) {
            alert('Error deleting user: ' + error.message);
        }
    };

    // Simulate API call (in a real app, this would be actual HTTP request)
    function simulateApiCall(endpoint, data, method = 'GET') {
        return new Promise((resolve) => {
            setTimeout(() => {
                // Simulate different responses based on endpoint
                if (endpoint.includes('/stats')) {
                    resolve({
                        success: true,
                        stats: {
                            total_users: 150,
                            today_attendance: 125,
                            total_attendance: 3250,
                            total_admins: 3
                        }
                    });
                } else if (endpoint.includes('/users')) {
                    // Simulate users data
                    const users = [
                        { id: 1, user_id: 'STU001', name: 'John Doe', user_type: 'student', class_name: 'Grade 10', section_name: 'A', roll_number: '001' },
                        { id: 2, user_id: 'TEA001', name: 'Jane Smith', user_type: 'teacher', class_name: null, section_name: null, roll_number: null },
                        { id: 3, user_id: 'STA001', name: 'Robert Johnson', user_type: 'staff', class_name: null, section_name: null, roll_number: null },
                        { id: 4, user_id: 'STU002', name: 'Emily Davis', user_type: 'student', class_name: 'Grade 9', section_name: 'B', roll_number: '015' },
                        { id: 5, user_id: 'TEA002', name: 'Michael Wilson', user_type: 'teacher', class_name: null, section_name: null, roll_number: null }
                    ];
                    resolve({ success: true, users: users });
                } else if (endpoint.includes('/daily-summary')) {
                    // Simulate attendance summary
                    const summary = [
                        { name: 'John Doe', user_id: 'STU001', type_name: 'student', first_scan: '2023-06-15T09:15:30Z', last_scan: '2023-06-15T15:45:20Z', is_late: false },
                        { name: 'Jane Smith', user_id: 'TEA001', type_name: 'teacher', first_scan: '2023-06-15T08:55:10Z', last_scan: '2023-06-15T16:30:45Z', is_late: false },
                        { name: 'Robert Johnson', user_id: 'STA001', type_name: 'staff', first_scan: '2023-06-15T09:05:22Z', last_scan: '2023-06-15T17:15:10Z', is_late: true }
                    ];
                    resolve({ success: true, summary: summary });
                } else if (endpoint.includes('/monthly-report')) {
                    // Simulate monthly report
                    const report = [
                        { name: 'John Doe', user_id: 'STU001', type_name: 'student', days_present: 25 },
                        { name: 'Jane Smith', user_id: 'TEA001', type_name: 'teacher', days_present: 22 },
                        { name: 'Robert Johnson', user_id: 'STA001', type_name: 'staff', days_present: 24 }
                    ];
                    resolve({ success: true, report: report });
                } else if (endpoint.includes('/logout')) {
                    resolve({ success: true, message: 'Logged out successfully' });
                } else if (endpoint.includes('/register')) {
                    resolve({ success: true, message: 'User registered successfully' });
                } else if (endpoint.includes('/users/')) {
                    if (method === 'DELETE') {
                        resolve({ success: true, message: 'User deleted successfully' });
                    }
                } else {
                    resolve({ success: true }); // Default success response
                }
            }, 500); // Simulate network delay
        });
    }
});