// Biometric Fingerprint Attendance System - Backend Server
const express = require('express');
const mysql = require('mysql2');
const bcrypt = require('bcrypt');
const cors = require('cors');
const path = require('path');
const session = require('express-session');
const crypto = require('crypto');
const jwt = require('jsonwebtoken');

// Import security modules
const { 
    hashPassword, 
    comparePassword, 
    SECURITY_CONFIG,
    encryptData,
    decryptData,
    generateToken,
    verifyToken
} = require('./security');

// Import security middleware
const {
    xssProtection,
    inputValidation,
    originValidation,
    authRateLimit,
    recordFailedAttempts,
    resetAttemptsOnSuccess,
    bodySizeLimit,
    deviceFingerprinting,
    sqlInjectionPrevention
} = require('./middleware/security');

const app = express();
const PORT = process.env.PORT || 3000;

// Security middleware
app.use(deviceFingerprinting);
app.use(xssProtection);
app.use(bodySizeLimit);
app.use(sqlInjectionPrevention);

// Regular middleware
app.use(cors());
app.use(express.json({ limit: '10mb' }));
app.use(express.urlencoded({ extended: true, limit: '10mb' }));

// Apply security middleware to specific routes
app.use('/api/admin/login', originValidation);
app.use('/api/admin/login', authRateLimit);
app.use('/api/admin/login', inputValidation);
app.use('/api/admin/login', recordFailedAttempts);
app.use('/api/admin/login', resetAttemptsOnSuccess);

app.use(session({
    secret: process.env.SESSION_SECRET || 'biometric_attendance_secret_key',
    resave: false,
    saveUninitialized: false,
    cookie: { secure: false, maxAge: 24 * 60 * 60 * 1000 } // 24 hours
}));

// Serve static files from frontend
app.use(express.static(path.join(__dirname, '../frontend')));

// Database connection
const db = mysql.createConnection({
    host: process.env.DB_HOST || 'localhost',
    user: process.env.DB_USER || 'root',
    password: process.env.DB_PASSWORD || '',
    database: process.env.DB_NAME || 'attendance_system'
});

// Connect to database
db.connect((err) => {
    if (err) {
        console.error('Database connection failed:', err);
        // In production, you might want to exit, but for demo we'll continue
        console.log('Running in demo mode without database...');
    } else {
        console.log('Connected to database');
    }
});

// Helper function to execute queries (with fallback for demo mode)
const executeQuery = (query, params = []) => {
    return new Promise((resolve, reject) => {
        if (!db._socket) {
            // Demo mode - return mock data
            console.log('Demo mode - returning mock data for query:', query);
            resolve({ rows: [], insertId: 1 });
            return;
        }
        
        db.query(query, params, (err, results) => {
            if (err) {
                console.error('Database query error:', err);
                reject(err);
            } else {
                resolve(results);
            }
        });
    });
};

// Middleware to check if user is admin
const requireAdmin = (req, res, next) => {
    // First check session
    if (req.session.adminId) {
        next();
        return;
    }
    
    // Then check for JWT token
    const authHeader = req.headers.authorization;
    if (!authHeader || !authHeader.startsWith('Bearer ')) {
        return res.status(401).json({ success: false, message: 'Unauthorized access' });
    }
    
    const token = authHeader.substring(7);
    
    try {
        const decoded = verifyToken(token);
        req.user = decoded;  // Attach user info to request
        next();
    } catch (error) {
        return res.status(401).json({ success: false, message: 'Invalid or expired token' });
    }
};

// API Routes

// Register new user
app.post('/api/register', async (req, res) => {
    try {
        const { id, name, type, class: className, section, rollNumber } = req.body;
        
        // Validation
        if (!id || !name || !type) {
            return res.status(400).json({ 
                success: false, 
                message: 'ID, name, and type are required' 
            });
        }
        
        // Get user type ID
        const userTypeResult = await executeQuery(
            'SELECT id FROM user_types WHERE type_name = ?', 
            [type]
        );
        
        if (userTypeResult.rows.length === 0) {
            return res.status(400).json({ 
                success: false, 
                message: 'Invalid user type' 
            });
        }
        
        const userTypeId = userTypeResult.rows[0].id;
        
        // Get class and section IDs if applicable
        let classId = null;
        let sectionId = null;
        
        if (type === 'student' && className) {
            const classResult = await executeQuery(
                'SELECT id FROM classes WHERE class_name = ?', 
                [className]
            );
            
            if (classResult.rows.length > 0) {
                classId = classResult.rows[0].id;
            }
        }
        
        if (type === 'student' && section && classId) {
            const sectionResult = await executeQuery(
                'SELECT id FROM sections WHERE section_name = ? AND class_id = ?', 
                [section, classId]
            );
            
            if (sectionResult.rows.length > 0) {
                sectionId = sectionResult.rows[0].id;
            }
        }
        
        // Check if user already exists
        const existingUser = await executeQuery(
            'SELECT id FROM users WHERE user_id = ?', 
            [id]
        );
        
        if (existingUser.rows.length > 0) {
            return res.status(400).json({ 
                success: false, 
                message: 'User ID already exists' 
            });
        }
        
        // Insert new user
        const insertUserResult = await executeQuery(
            'INSERT INTO users (user_id, name, user_type_id, class_id, section_id, roll_number) VALUES (?, ?, ?, ?, ?, ?)', 
            [id, name, userTypeId, classId, sectionId, rollNumber]
        );
        
        res.status(200).json({ 
            success: true, 
            message: 'User registered successfully',
            userId: insertUserResult.insertId
        });
        
    } catch (error) {
        console.error('Registration error:', error);
        res.status(500).json({ 
            success: false, 
            message: 'Server error during registration' 
        });
    }
});

// Mark attendance
app.post('/api/attendance/mark', requireAdmin, async (req, res) => {
    try {
        const { userId } = req.body;
        
        if (!userId) {
            return res.status(400).json({ 
                success: false, 
                message: 'User ID is required' 
            });
        }
        
        // Check if user exists
        const userCheck = await executeQuery(
            'SELECT id FROM users WHERE user_id = ?', 
            [userId]
        );
        
        if (userCheck.rows.length === 0) {
            return res.status(404).json({ 
                success: false, 
                message: 'User not found' 
            });
        }
        
        // Check if attendance already marked today
        const today = new Date().toISOString().slice(0, 10);
        const existingAttendance = await executeQuery(
            `SELECT id FROM attendance 
             WHERE user_id = ? AND DATE(scan_time) = ?`, 
            [userId, today]
        );
        
        if (existingAttendance.rows.length > 0) {
            return res.status(400).json({ 
                success: false, 
                message: 'Attendance already marked today' 
            });
        }
        
        // Insert attendance record
        const insertAttendanceResult = await executeQuery(
            'INSERT INTO attendance (user_id, verification_method) VALUES (?, ?)', 
            [userId, 'fingerprint']
        );
        
        res.status(200).json({ 
            success: true, 
            message: 'Attendance marked successfully',
            attendanceId: insertAttendanceResult.insertId
        });
        
    } catch (error) {
        console.error('Attendance marking error:', error);
        res.status(500).json({ 
            success: false, 
            message: 'Server error during attendance marking' 
        });
    }
});

// Admin login
app.post('/api/admin/login', async (req, res) => {
    try {
        const { username, password } = req.body;
        
        if (!username || !password) {
            return res.status(400).json({ 
                success: false, 
                message: 'Username and password are required' 
            });
        }
        
        // Validate inputs
        if (!validateInput(username) || !validateInput(password, 128)) {
            return res.status(400).json({ 
                success: false, 
                message: 'Invalid input format' 
            });
        }
        
        // In demo mode, accept hardcoded admin credentials
        if (!db._socket) {
            if (username === 'admin' && password === 'admin') {
                // Generate JWT token for demo
                const token = generateToken({ 
                    id: 1, 
                    username: username, 
                    role: 'admin' 
                });
                
                req.session.adminId = 1;
                req.session.username = username;
                return res.status(200).json({ 
                    success: true, 
                    message: 'Login successful', 
                    token: token,
                    user: { id: 1, username: username }
                });
            } else {
                return res.status(401).json({ 
                    success: false, 
                    message: 'Invalid credentials' 
                });
            }
        }
        
        // In real mode, check database
        const adminResult = await executeQuery(
            'SELECT id, username, password_hash FROM admins WHERE username = ?', 
            [username]
        );
        
        if (adminResult.rows.length === 0) {
            return res.status(401).json({ 
                success: false, 
                message: 'Invalid credentials' 
            });
        }
        
        const admin = adminResult.rows[0];
        const isValidPassword = await comparePassword(password, admin.password_hash);
        
        if (!isValidPassword) {
            return res.status(401).json({ 
                success: false, 
                message: 'Invalid credentials' 
            });
        }
        
        // Generate JWT token
        const token = generateToken({ 
            id: admin.id, 
            username: admin.username, 
            role: 'admin' 
        });
        
        // Set session
        req.session.adminId = admin.id;
        req.session.username = admin.username;
        
        res.status(200).json({ 
            success: true, 
            message: 'Login successful',
            token: token,
            user: { id: admin.id, username: admin.username }
        });
        
    } catch (error) {
        console.error('Login error:', error);
        res.status(500).json({ 
            success: false, 
            message: 'Server error during login' 
        });
    }
});

// Admin logout
app.post('/api/admin/logout', (req, res) => {
    req.session.destroy((err) => {
        if (err) {
            console.error('Session destroy error:', err);
        }
        res.status(200).json({ success: true, message: 'Logged out successfully' });
    });
});

// Get all users (admin only)
app.get('/api/users', requireAdmin, async (req, res) => {
    try {
        const users = await executeQuery(`
            SELECT 
                u.id,
                u.user_id,
                u.name,
                ut.type_name as user_type,
                c.class_name,
                s.section_name,
                u.roll_number,
                u.created_at
            FROM users u
            JOIN user_types ut ON u.user_type_id = ut.id
            LEFT JOIN classes c ON u.class_id = c.id
            LEFT JOIN sections s ON u.section_id = s.id
            ORDER BY u.created_at DESC
        `);
        
        res.status(200).json({ 
            success: true, 
            users: users.rows || [] 
        });
        
    } catch (error) {
        console.error('Get users error:', error);
        res.status(500).json({ 
            success: false, 
            message: 'Server error while fetching users' 
        });
    }
});

// Get attendance for a specific user
app.get('/api/attendance/:userId', requireAdmin, async (req, res) => {
    try {
        const { userId } = req.params;
        
        const attendanceRecords = await executeQuery(`
            SELECT 
                id,
                scan_time,
                device_id,
                verification_method,
                is_late,
                note
            FROM attendance 
            WHERE user_id = ? 
            ORDER BY scan_time DESC 
            LIMIT 50
        `, [userId]);
        
        res.status(200).json({ 
            success: true, 
            attendance: attendanceRecords.rows || [] 
        });
        
    } catch (error) {
        console.error('Get attendance error:', error);
        res.status(500).json({ 
            success: false, 
            message: 'Server error while fetching attendance' 
        });
    }
});

// Get daily attendance summary
app.get('/api/attendance/daily-summary', requireAdmin, async (req, res) => {
    try {
        const { date = new Date().toISOString().split('T')[0] } = req.query;
        
        const summary = await executeQuery(`
            SELECT 
                u.name,
                u.user_id,
                ut.type_name,
                c.class_name,
                s.section_name,
                MIN(a.scan_time) as first_scan,
                MAX(a.scan_time) as last_scan,
                COUNT(*) as total_scans
            FROM attendance a
            JOIN users u ON a.user_id = u.user_id
            JOIN user_types ut ON u.user_type_id = ut.id
            LEFT JOIN classes c ON u.class_id = c.id
            LEFT JOIN sections s ON u.section_id = s.id
            WHERE DATE(a.scan_time) = ?
            GROUP BY u.user_id
            ORDER BY first_scan ASC
        `, [date]);
        
        res.status(200).json({ 
            success: true, 
            summary: summary.rows || [],
            date: date
        });
        
    } catch (error) {
        console.error('Daily summary error:', error);
        res.status(500).json({ 
            success: false, 
            message: 'Server error while fetching daily summary' 
        });
    }
});

// Get monthly attendance report
app.get('/api/attendance/monthly-report', requireAdmin, async (req, res) => {
    try {
        const { month, year } = req.query;
        const currentYear = year || new Date().getFullYear();
        const currentMonth = month || String(new Date().getMonth() + 1).padStart(2, '0');
        
        const report = await executeQuery(`
            SELECT 
                u.name,
                u.user_id,
                ut.type_name,
                c.class_name,
                s.section_name,
                u.roll_number,
                COUNT(*) as days_present
            FROM attendance a
            JOIN users u ON a.user_id = u.user_id
            JOIN user_types ut ON u.user_type_id = ut.id
            LEFT JOIN classes c ON u.class_id = c.id
            LEFT JOIN sections s ON u.section_id = s.id
            WHERE YEAR(a.scan_time) = ? AND MONTH(a.scan_time) = ?
            GROUP BY u.user_id
            ORDER BY u.name ASC
        `, [currentYear, currentMonth]);
        
        res.status(200).json({ 
            success: true, 
            report: report.rows || [],
            month: currentMonth,
            year: currentYear
        });
        
    } catch (error) {
        console.error('Monthly report error:', error);
        res.status(500).json({ 
            success: false, 
            message: 'Server error while fetching monthly report' 
        });
    }
});

// Delete user (admin only)
app.delete('/api/users/:userId', requireAdmin, async (req, res) => {
    try {
        const { userId } = req.params;
        
        // Delete user (this will also delete related attendance and fingerprint records due to foreign key constraints)
        await executeQuery('DELETE FROM users WHERE user_id = ?', [userId]);
        
        res.status(200).json({ 
            success: true, 
            message: 'User deleted successfully' 
        });
        
    } catch (error) {
        console.error('Delete user error:', error);
        res.status(500).json({ 
            success: false, 
            message: 'Server error while deleting user' 
        });
    }
});

// Get system statistics
app.get('/api/stats', requireAdmin, async (req, res) => {
    try {
        const stats = await executeQuery(`
            SELECT 
                (SELECT COUNT(*) FROM users) as total_users,
                (SELECT COUNT(*) FROM attendance WHERE DATE(scan_time) = CURDATE()) as today_attendance,
                (SELECT COUNT(*) FROM attendance) as total_attendance,
                (SELECT COUNT(*) FROM admins) as total_admins
        `);
        
        res.status(200).json({ 
            success: true, 
            stats: stats.rows[0] || {} 
        });
        
    } catch (error) {
        console.error('Stats error:', error);
        res.status(500).json({ 
            success: false, 
            message: 'Server error while fetching statistics' 
        });
    }
});

// Health check endpoint
app.get('/health', (req, res) => {
    res.status(200).json({ 
        status: 'OK', 
        timestamp: new Date().toISOString(),
        uptime: process.uptime()
    });
});

// Serve the main page
app.get('/', (req, res) => {
    res.sendFile(path.join(__dirname, '../frontend/index.html'));
});

// Start server
app.listen(PORT, () => {
    console.log(`Server running on http://localhost:${PORT}`);
    console.log('Biometric Attendance System API is ready!');
});

module.exports = app;