// Security Configuration for Biometric Attendance System
const bcrypt = require('bcrypt');
const crypto = require('crypto');
const jwt = require('jsonwebtoken'); // We'll add this to package.json later

// Security configuration constants
const SECURITY_CONFIG = {
    // Password hashing
    SALT_ROUNDS: 12,
    
    // JWT configuration
    JWT_SECRET: process.env.JWT_SECRET || 'biometric_attendance_jwt_secret_key',
    JWT_EXPIRES_IN: '24h', // Token expires in 24 hours
    
    // Session configuration
    SESSION_SECRET: process.env.SESSION_SECRET || 'biometric_attendance_session_secret',
    SESSION_MAX_AGE: 24 * 60 * 60 * 1000, // 24 hours
    
    // Encryption settings
    ENCRYPTION_ALGORITHM: 'aes-256-gcm',
    ENCRYPTION_KEY: process.env.ENCRYPTION_KEY || crypto.randomBytes(32), // 32 bytes for AES-256
    IV_LENGTH: 16, // Initialization vector length for AES
    
    // Rate limiting
    MAX_LOGIN_ATTEMPTS: 5,
    LOCKOUT_TIME: 15 * 60 * 1000, // 15 minutes
    
    // Input validation
    MAX_INPUT_LENGTH: 100,
    ALLOWED_FILE_TYPES: ['image/bmp', 'image/png', 'image/jpeg'],
    MAX_FILE_SIZE: 5 * 1024 * 1024, // 5MB
    
    // Security headers
    SECURITY_HEADERS: {
        'X-Content-Type-Options': 'nosniff',
        'X-Frame-Options': 'DENY',
        'X-XSS-Protection': '1; mode=block',
        'Strict-Transport-Security': 'max-age=31536000; includeSubDomains',
        'Content-Security-Policy': "default-src 'self'; script-src 'self' 'unsafe-inline'; style-src 'self' 'unsafe-inline'"
    }
};

// Utility functions for security operations

// Hash password using bcrypt
async function hashPassword(password) {
    try {
        const saltRounds = SECURITY_CONFIG.SALT_ROUNDS;
        const hashedPassword = await bcrypt.hash(password, saltRounds);
        return hashedPassword;
    } catch (error) {
        throw new Error('Password hashing failed');
    }
}

// Compare password with hash
async function comparePassword(password, hash) {
    try {
        const isMatch = await bcrypt.compare(password, hash);
        return isMatch;
    } catch (error) {
        throw new Error('Password comparison failed');
    }
}

// Encrypt sensitive data (like fingerprint templates)
function encryptData(data) {
    try {
        const iv = crypto.randomBytes(SECURITY_CONFIG.IV_LENGTH);
        const cipher = crypto.createCipher(SECURITY_CONFIG.ENCRYPTION_ALGORITHM, SECURITY_CONFIG.ENCRYPTION_KEY);
        cipher.setAAD(Buffer.from('biometric_data')); // Additional authenticated data
        
        let encrypted = cipher.update(data, 'utf8', 'hex');
        encrypted += cipher.final('hex');
        
        const authTag = cipher.getAuthTag();
        
        return {
            encryptedData: encrypted,
            iv: iv.toString('hex'),
            authTag: authTag.toString('hex')
        };
    } catch (error) {
        throw new Error('Data encryption failed');
    }
}

// Decrypt sensitive data
function decryptData(encryptedData, iv, authTag) {
    try {
        const decipher = crypto.createDecipher(SECURITY_CONFIG.ENCRYPTION_ALGORITHM, SECURITY_CONFIG.ENCRYPTION_KEY);
        decipher.setAAD(Buffer.from('biometric_data'));
        decipher.setAuthTag(Buffer.from(authTag, 'hex'));
        
        let decrypted = decipher.update(encryptedData, 'hex', 'utf8');
        decrypted += decipher.final('utf8');
        
        return decrypted;
    } catch (error) {
        throw new Error('Data decryption failed');
    }
}

// Generate secure random token
function generateSecureToken(length = 32) {
    return crypto.randomBytes(length).toString('hex');
}

// Validate user input
function validateInput(input, maxLength = SECURITY_CONFIG.MAX_INPUT_LENGTH) {
    if (!input || typeof input !== 'string') {
        return false;
    }
    
    if (input.length > maxLength) {
        return false;
    }
    
    // Check for potentially malicious patterns
    const maliciousPatterns = [
        /<script/i,
        /javascript:/i,
        /vbscript:/i,
        /on\w+\s*=/i,
        /<iframe/i,
        /<object/i,
        /<embed/i
    ];
    
    for (const pattern of maliciousPatterns) {
        if (pattern.test(input)) {
            return false;
        }
    }
    
    return true;
}

// Sanitize user input
function sanitizeInput(input) {
    if (typeof input !== 'string') {
        return input;
    }
    
    // Remove potentially dangerous characters/patterns
    return input
        .replace(/<script\b[^<]*(?:(?!<\/script>)<[^<]*)*<\/script>/gi, '')
        .replace(/<iframe\b[^<]*(?:(?!<\/iframe>)<[^<]*)*<\/iframe>/gi, '')
        .replace(/<object\b[^<]*(?:(?!<\/object>)<[^<]*)*<\/object>/gi, '')
        .replace(/<embed\b[^<]*(?:(?!<\/embed>)<[^<]*)*<\/embed>/gi, '')
        .replace(/<form\b[^<]*(?:(?!<\/form>)<[^<]*)*<\/form>/gi, '')
        .trim();
}

// Generate JWT token
function generateToken(payload) {
    return jwt.sign(payload, SECURITY_CONFIG.JWT_SECRET, {
        expiresIn: SECURITY_CONFIG.JWT_EXPIRES_IN
    });
}

// Verify JWT token
function verifyToken(token) {
    try {
        return jwt.verify(token, SECURITY_CONFIG.JWT_SECRET);
    } catch (error) {
        throw new Error('Invalid or expired token');
    }
}

// Rate limiting utility (in-memory store - use Redis in production)
const loginAttempts = new Map();

function recordFailedAttempt(username) {
    const now = Date.now();
    const attempts = loginAttempts.get(username) || { count: 0, lastAttempt: now };
    
    attempts.count++;
    attempts.lastAttempt = now;
    
    loginAttempts.set(username, attempts);
    
    return attempts.count >= SECURITY_CONFIG.MAX_LOGIN_ATTEMPTS;
}

function isAccountLocked(username) {
    const attempts = loginAttempts.get(username);
    
    if (!attempts) {
        return false;
    }
    
    // Check if enough time has passed since last attempt
    if (Date.now() - attempts.lastAttempt > SECURITY_CONFIG.LOCKOUT_TIME) {
        loginAttempts.delete(username); // Clean up after lockout period
        return false;
    }
    
    return attempts.count >= SECURITY_CONFIG.MAX_LOGIN_ATTEMPTS;
}

function resetLoginAttempts(username) {
    loginAttempts.delete(username);
}

// Apply security headers to response
function applySecurityHeaders(res) {
    Object.entries(SECURITY_CONFIG.SECURITY_HEADERS).forEach(([header, value]) => {
        res.setHeader(header, value);
    });
}

// Check if request is coming from a valid source (CSRF protection)
function isValidOrigin(req) {
    const origin = req.headers.origin;
    const referer = req.headers.referer;
    
    // In production, define your allowed origins
    const allowedOrigins = [
        'http://localhost:3000',
        'http://localhost:3001',
        'http://127.0.0.1:3000',
        process.env.ALLOWED_ORIGIN || ''
    ].filter(Boolean); // Remove empty strings
    
    return allowedOrigins.some(allowed => 
        origin?.startsWith(allowed) || referer?.startsWith(allowed)
    );
}

// Generate fingerprint hash for device identification
function generateDeviceFingerprint(req) {
    const userAgent = req.headers['user-agent'] || '';
    const acceptLanguage = req.headers['accept-language'] || '';
    const acceptEncoding = req.headers['accept-encoding'] || '';
    const accept = req.headers['accept'] || '';
    
    const fingerprintString = `${userAgent}${acceptLanguage}${acceptEncoding}${accept}`;
    return crypto.createHash('sha256').update(fingerprintString).digest('hex');
}

// Export security utilities
module.exports = {
    SECURITY_CONFIG,
    hashPassword,
    comparePassword,
    encryptData,
    decryptData,
    generateSecureToken,
    validateInput,
    sanitizeInput,
    generateToken,
    verifyToken,
    recordFailedAttempt,
    isAccountLocked,
    resetLoginAttempts,
    applySecurityHeaders,
    isValidOrigin,
    generateDeviceFingerprint
};