// Security Middleware for Biometric Attendance System
const { 
    validateInput, 
    sanitizeInput, 
    applySecurityHeaders, 
    isValidOrigin, 
    generateDeviceFingerprint,
    isAccountLocked,
    recordFailedAttempt,
    resetLoginAttempts
} = require('../security');

// XSS Protection Middleware
function xssProtection(req, res, next) {
    // Apply security headers
    applySecurityHeaders(res);
    
    // Sanitize common input fields
    if (req.body) {
        for (const key in req.body) {
            if (typeof req.body[key] === 'string') {
                req.body[key] = sanitizeInput(req.body[key]);
            }
        }
    }
    
    next();
}

// Input Validation Middleware
function inputValidation(req, res, next) {
    // Validate common fields
    if (req.body.username && !validateInput(req.body.username)) {
        return res.status(400).json({
            success: false,
            message: 'Invalid username format'
        });
    }
    
    if (req.body.password && !validateInput(req.body.password, 128)) {
        return res.status(400).json({
            success: false,
            message: 'Invalid password format'
        });
    }
    
    if (req.body.email && !validateInput(req.body.email)) {
        return res.status(400).json({
            success: false,
            message: 'Invalid email format'
        });
    }
    
    next();
}

// CORS and Origin Validation Middleware
function originValidation(req, res, next) {
    // Check if request is coming from a valid origin
    if (!isValidOrigin(req)) {
        return res.status(403).json({
            success: false,
            message: 'Request from invalid origin'
        });
    }
    
    // Set CORS headers
    res.header('Access-Control-Allow-Origin', '*');
    res.header('Access-Control-Allow-Methods', 'GET, POST, PUT, DELETE, OPTIONS');
    res.header('Access-Control-Allow-Headers', 'Origin, X-Requested-With, Content-Type, Accept, Authorization');
    
    next();
}

// Rate Limiting Middleware for Authentication
function authRateLimit(req, res, next) {
    if (req.path.includes('/login') || req.path.includes('/register')) {
        const username = req.body.username || req.query.username;
        
        if (username && isAccountLocked(username)) {
            return res.status(429).json({
                success: false,
                message: 'Account temporarily locked due to multiple failed login attempts. Please try again later.'
            });
        }
    }
    
    next();
}

// Failed Attempt Recording Middleware
function recordFailedAttempts(req, res, next) {
    // Capture the original res.json method
    const originalJson = res.json;
    
    res.json = function(data) {
        // If this is a login attempt that failed
        if ((req.path.includes('/login') || req.path.includes('/auth')) && 
            data && typeof data === 'object' && data.success === false) {
            
            const username = req.body.username || req.query.username;
            if (username) {
                const isLocked = recordFailedAttempt(username);
                
                if (isLocked) {
                    // Override response if account is now locked
                    return originalJson.call(this, {
                        success: false,
                        message: 'Account temporarily locked due to multiple failed login attempts. Please try again later.'
                    });
                }
            }
        }
        
        // Call original json method
        originalJson.call(this, data);
    };
    
    next();
}

// Successful Login Reset Middleware
function resetAttemptsOnSuccess(req, res, next) {
    // Capture the original res.json method
    const originalJson = res.json;
    
    res.json = function(data) {
        // If login was successful, reset failed attempts
        if ((req.path.includes('/login') || req.path.includes('/auth')) && 
            data && typeof data === 'object' && data.success === true) {
            
            const username = req.body.username || req.query.username;
            if (username) {
                resetLoginAttempts(username);
            }
        }
        
        // Call original json method
        originalJson.call(this, data);
    };
    
    next();
}

// Request Body Size Limiting Middleware
function bodySizeLimit(req, res, next) {
    const contentLength = req.headers['content-length'];
    
    if (contentLength && parseInt(contentLength) > 10 * 1024 * 1024) { // 10MB limit
        return res.status(413).json({
            success: false,
            message: 'Request body too large'
        });
    }
    
    next();
}

// Device Fingerprinting Middleware
function deviceFingerprinting(req, res, next) {
    req.deviceFingerprint = generateDeviceFingerprint(req);
    next();
}

// SQL Injection Prevention Middleware (Basic)
function sqlInjectionPrevention(req, res, next) {
    if (req.query) {
        for (const key in req.query) {
            const value = req.query[key];
            if (typeof value === 'string') {
                // Check for SQL injection patterns
                if (/(DROP|DELETE|INSERT|UPDATE|SELECT|UNION|EXEC|CALL|ALTER|CREATE|GRANT|REVOKE|TRUNCATE)/i.test(value)) {
                    return res.status(400).json({
                        success: false,
                        message: 'Potential SQL injection detected'
                    });
                }
            }
        }
    }
    
    if (req.body) {
        for (const key in req.body) {
            const value = req.body[key];
            if (typeof value === 'string') {
                // Check for SQL injection patterns
                if (/(DROP|DELETE|INSERT|UPDATE|SELECT|UNION|EXEC|CALL|ALTER|CREATE|GRANT|REVOKE|TRUNCATE)/i.test(value)) {
                    return res.status(400).json({
                        success: false,
                        message: 'Potential SQL injection detected'
                    });
                }
            }
        }
    }
    
    next();
}

// Export all middleware functions
module.exports = {
    xssProtection,
    inputValidation,
    originValidation,
    authRateLimit,
    recordFailedAttempts,
    resetAttemptsOnSuccess,
    bodySizeLimit,
    deviceFingerprinting,
    sqlInjectionPrevention
};